/*:
 * @target MZ
 * @plugindesc [MEG] UI/入力の復帰保険（ロード後・タイトル復帰後 + コモン予約解除） v1.1.0
 * @author HS
 *
 * @param enableOnLoadSuccess
 * @type boolean
 * @default true
 *
 * @param enableOnGotoTitle
 * @type boolean
 * @default true
 *
 * @param enableOnAnySceneGoto
 * @type boolean
 * @default false
 *
 * @param resetCommonEventLocks
 * @text コモンイベント予約/キューを解除
 * @type boolean
 * @default true
 *
 * @param resetInterpreter
 * @text マップインタプリタをクリア（存在時）
 * @type boolean
 * @default true
 *
 * @param callSceneMethod
 * @type string
 * @default
 *
 * @param debugLog
 * @type boolean
 * @default false
 *
 * @command ForceRecoverNow
 * @text その場で復帰処理を実行
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "MEG_UIRecoverAfterLoad_Fix";
  const p = PluginManager.parameters(PLUGIN_NAME);

  const enableOnLoadSuccess  = String(p.enableOnLoadSuccess) === "true";
  const enableOnGotoTitle    = String(p.enableOnGotoTitle) === "true";
  const enableOnAnySceneGoto = String(p.enableOnAnySceneGoto) === "true";

  const resetCommonEventLocks = String(p.resetCommonEventLocks) === "true";
  const resetInterpreter      = String(p.resetInterpreter) === "true";

  const callSceneMethod = String(p.callSceneMethod || "").trim();
  const debugLog = String(p.debugLog) === "true";

  let _recoverRequested = false;
  let _recoverReason = "";

  function log(...args) {
    if (debugLog) console.log(`[${PLUGIN_NAME}]`, ...args);
  }

  function requestRecover(reason) {
    _recoverRequested = true;
    _recoverReason = reason || "unknown";
    log("Recover requested:", _recoverReason);
  }

  function clearCommonEventLocks() {
    try {
      if (window.$gameTemp) {
        // 予約解除（標準に存在）
        if (typeof $gameTemp.clearCommonEventReservation === "function") {
          $gameTemp.clearCommonEventReservation();
        } else if ("_commonEventId" in $gameTemp) {
          $gameTemp._commonEventId = 0;
        }

        // キュー運用している場合の保険（ButtonPictureが参照）
        if (Array.isArray($gameTemp._commonEventQueue)) {
          $gameTemp._commonEventQueue.length = 0;
        }
      }
    } catch (e) {
      console.error(`[${PLUGIN_NAME}] clearCommonEventLocks failed:`, e);
    }
  }

  function clearInterpreterIfAny() {
    try {
      if (resetInterpreter && window.$gameMap && $gameMap._interpreter && typeof $gameMap._interpreter.clear === "function") {
        $gameMap._interpreter.clear();
      }
    } catch (e) {
      console.error(`[${PLUGIN_NAME}] clearInterpreterIfAny failed:`, e);
    }
  }

  function doRecoverForScene(scene, reason) {
    try {
      Input.clear();
      TouchInput.clear();

      if (resetCommonEventLocks) clearCommonEventLocks();
      clearInterpreterIfAny();

      if (scene && callSceneMethod && typeof scene[callSceneMethod] === "function") {
        log(`Calling scene method: ${callSceneMethod}()`);
        scene[callSceneMethod]();
      }

      log("Recovered on scene:", scene?.constructor?.name, "reason:", reason);
    } catch (e) {
      console.error(`[${PLUGIN_NAME}] Recover failed:`, e);
    }
  }

  PluginManager.registerCommand(PLUGIN_NAME, "ForceRecoverNow", () => {
    doRecoverForScene(SceneManager._scene, "pluginCommand");
  });

  // ロード成功後に復帰要求
  if (enableOnLoadSuccess && Scene_Load && Scene_Load.prototype.onLoadSuccess) {
    const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
      _onLoadSuccess.call(this);
      requestRecover("loadSuccess");
    };
  }

  // シーン遷移時の復帰要求
  const _goto = SceneManager.goto;
  SceneManager.goto = function(sceneClass) {
    const current = SceneManager._scene;
    const currentName = current?.constructor?.name || "";
    const nextName = sceneClass?.name || "";

    if (enableOnAnySceneGoto) {
      requestRecover(`goto:${currentName}->${nextName}`);
    } else if (enableOnGotoTitle && sceneClass === Scene_Title) {
      requestRecover(`gotoTitle:${currentName}->${nextName}`);
    }

    _goto.call(this, sceneClass);
  };

  // シーン開始時に一回だけ実行
  const _start = Scene_Base.prototype.start;
  Scene_Base.prototype.start = function() {
    _start.call(this);

    if (_recoverRequested) {
      const reason = _recoverReason;
      _recoverRequested = false;
      _recoverReason = "";
      doRecoverForScene(this, reason);
    }
  };
})();
